/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        $Id: CLDAPKeyServer.cpp,v 1.17 2000/07/10 22:05:00 dallen Exp $
____________________________________________________________________________*/

#include <string.h>
#include <ctype.h>
/* <stdlib.h> includs <sys/time.h> which aCC barfs on if <sys/sigevent.h>
   in not included. */
#if PGP_COMPILER_HPUX
#include <sys/sigevent.h>
#endif /* PGP_COMPILER_HPUX */
#include <stdlib.h>

#include "pgpStrings.h"
#include "pgpOptionList.h"
#include "pgpEventPriv.h"
#include "pgpKeys.h"
#include "pgpFeatures.h"
#include "pgpSockets.h"
#include "pgpMem.h"
#include "pgpMemoryMgr.h"
#include "pgpDebug.h"

#include "CLDAPKeyServer.h"



#define ThrowIfLDAPCanceled_()	if (mCanceled) \
									ThrowPGPError_(kPGPError_UserAbort);
#define ThrowIfPGPErrorOrLDAPCanceled_(x)	{	ThrowIfPGPError_(x); \
												ThrowIfLDAPCanceled_(); }

static const char *	kBaseKeyspaceDN = "baseKeyspaceDN";
static const char *	kBasePendingDN = "basePendingDN";
static const char *	kVersionDN = "version";
static const char *	kPGPServerInfoCN = "cn=PGPServerInfo";
static const char *	kMonitorCN = "cn=monitor";
static const char *	kObjectClassAny = "(objectclass=*)";
static const char *	kPGPKeyAttribute = "pgpKey";
static const char *	kPGPKeyV2Attribute = "pgpKeyV2";
static const char *	kRequestPGPRequest = "PGPrequest";
static const char *	kDefaultPGPGroupspace = "ou=default,o=pgp groupspace,c=us";
//static const char *	kPGPSrvGroupFile = "pgpSrvGroupFile";
static const char *	kPGPSrvGroupFileV2 = "pgpSrvGroupFileV2";
static const char *	kRequestRequest = "Request: ";
static const char * kRequestLocation = "Location: ";
static const char * kRequestPending = "pending";
static const char *	kRequestActive = "active";
static const char *	kAddAndRequestDNPrefix = "pgpcertid=virtual, ";
static const char *	kAdd = "add";
static const char *	kDelete = "delete";
static const char *	kDisable = "disable";
static const char *	kObjectClass = "objectclass";
static const char * kPGPSrvGroupFileClass = "pgpsrvgroupfile";



CLDAPKeyServer::CLDAPKeyServer(
	PGPContextRef			inContext,
	const char *			inHostName,
	PGPUInt32				inHostAddress,
	PGPUInt16				inHostPort,
	PGPKeyServerProtocol	inType,
	PGPKeyServerClass		inClass,
	PGPKeyServerAccessType	inAccessType,
	PGPKeyServerKeySpace	inKeySpace)
	: CKeyServer(inContext, inHostName, inHostAddress, inHostPort, 0, inType, inClass),
	  mContext(inContext),
	  mAccessType(inAccessType), mKeySpace(inKeySpace), mLDAP(0),
	  mBaseKeySpaceDN(0), mBasePendingDN(0),
	  mExportFormat(kPGPExportFormat_Basic)
{
}



CLDAPKeyServer::~CLDAPKeyServer()
{
}



	void
CLDAPKeyServer::Open(
	PGPtlsSessionRef	inTLSSession)
{
	StPreserveSocketsEventHandler	preserve(this);
	PGPldapMessageRef				message = kInvalidPGPldapMessageRef;
	PGPldapResult					result;
	char **							value = 0;
	char *							szError = NULL;
	PGPSocketRef					sock;
	PGPError						pgpError	= kPGPError_NoErr;

	SetErrorString(0);
	try
	{
		PGPldapMessageRef	nextMessage		= kInvalidPGPldapMessageRef;
		char *				attribute		= NULL;
		PGPberElementRef	iterAttribute	= kInvalidPGPberElementRef;
		static const char *	attr[4] = { kBaseKeyspaceDN,
									kBasePendingDN, kVersionDN, 0 };

		CKeyServer::Open(inTLSSession);
		
		// Open ldap connection
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Opening);
		ThrowIfPGPErrorOrLDAPCanceled_(pgpError);

		pgpError = PGPNewLDAPContext( mContext, &mLDAP );

		if (mLDAP == kInvalidPGPldapContextRef)
			ThrowPGPError_(kPGPError_ServerOpenFailed);
		
		pgpError = PGPldapOpen( mLDAP, mHostName, mHostPort );
		ThrowIfLDAPCanceled_();

		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );
		
		// Secure connection for LDAPS
		if( mType == kPGPKeyServerProtocol_LDAPS)
		{
			pgpError = PGPldapGetOption( mLDAP, kPGPldapOpt_Desc, &sock );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			if( ( mTLSSession == kInvalidPGPtlsSessionRef ) || 
				( PGPSocketsEstablishTLSSession(
					(PGPSocketRef) sock, mTLSSession ) 
				!= kPGPError_NoErr ) )
			{
				pgpError = pgpEventKeyServerTLS(
								mContext,
								mEventHandler,
								mEventHandlerData,
								(PGPKeyServerRef) this,
								kPGPKeyServerState_TLSUnableToSecureConnection,
								mTLSSession );
			}
			else
			{
				pgpError = pgpEventKeyServerTLS(
								mContext,
								mEventHandler,
								mEventHandlerData,
								(PGPKeyServerRef) this,
								kPGPKeyServerState_TLSConnectionSecured,
								mTLSSession );

				mSecured = true;
			}
			ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
		}
		
		// Retrieve key space distinguished names

		pgpError = PGPNewLDAPMessage( mLDAP, &message );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		pgpError = PGPldapSearchSync( mLDAP,
									  (char *) kPGPServerInfoCN,
									  kPGPldapScope_Base,
									  (char *) kObjectClassAny,
									  (char **) attr,
									  FALSE,
									  message );
		ThrowIfLDAPCanceled_();
		if( IsPGPError( pgpError ) )
			ThrowPGPError_(kPGPError_ServerOpenFailed);

		pgpError = PGPldapFirstEntry( mLDAP, message, &nextMessage );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		while (nextMessage != NULL)
		{
			pgpError = PGPldapFirstAttribute( mLDAP,
								   nextMessage,
								   &iterAttribute,
								   &attribute );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			while (attribute != NULL)
			{
				pgpError = PGPldapGetValues( mLDAP,
								  nextMessage,
								  attribute,
								  &value );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerOpenFailed );

				if ((value != 0) && (*value != 0))
				{
					if (pgpCompareStringsIgnoreCase(attribute,
							attr[0]) == 0) 
					{
						mBaseKeySpaceDN = new char[strlen(*value) + 1];
						if (mBaseKeySpaceDN == 0)
							ThrowPGPError_(kPGPError_OutOfMemory);

						strcpy(mBaseKeySpaceDN, *value);
					} 
					else if (pgpCompareStringsIgnoreCase(attribute,
								attr[1]) == 0)
					{
						mBasePendingDN = new char[strlen(*value) + 1];
						if (mBasePendingDN == 0)
							ThrowPGPError_(kPGPError_OutOfMemory);

						strcpy(mBasePendingDN, *value);
					}
					else if (pgpCompareStringsIgnoreCase(attribute,
								attr[2]) == 0)
					{
						if (atol(*value) > 1)
							mExportFormat = kPGPExportFormat_Complete;

					}
					(void) PGPFreeLDAPValues( value );
					value = 0;
				}
				else 
				{
					pgpError = PGPldapGetErrno( mLDAP, NULL, NULL, &result );
					if( IsPGPError( pgpError ) )
						ThrowPGPError_( kPGPError_ServerOpenFailed );

					if (( value == 0 ) && ( result != kPGPldapResult_Success ))
					{
						if( result == kPGPldapResult_InsufficientAccess )
							ThrowPGPError_(kPGPError_ServerAuthorizationFailed);
						else
							ThrowPGPError_(kPGPError_ServerOpenFailed);
					}
				}
				(void) PGPFreeData( attribute );
				attribute = NULL;

				pgpError = PGPldapNextAttribute( mLDAP,
									  nextMessage,
									  iterAttribute,
									  &attribute );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerOpenFailed );
			}

			pgpError = PGPldapGetErrno( mLDAP, NULL, NULL, &result );
			if( IsPGPError( pgpError ) || ( result != kPGPldapResult_Success ) )
				ThrowPGPError_(kPGPError_ServerOpenFailed);

			pgpError = PGPldapNextEntry( mLDAP,
							  nextMessage,
							  &nextMessage );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );
		}
		if ( (mBaseKeySpaceDN == 0) || (mBasePendingDN == 0) )
			ThrowPGPError_(kPGPError_ServerOpenFailed);

		PGPFreeLDAPMessage( message );
		message = kInvalidPGPldapMessageRef;
	}
	
	catch (...) {
		delete[] mBaseKeySpaceDN;
		mBaseKeySpaceDN = 0;
		delete[] mBasePendingDN;
		mBasePendingDN = 0;
		
		if( IsntNull( value ) )
			PGPFreeLDAPValues( value );

		if( PGPldapMessageRefIsValid( message ) )
			PGPFreeLDAPMessage( message );

		mTLSSession = kInvalidPGPtlsSessionRef;
		mIsOpen = false;
		mSecured = false;
		if( PGPldapContextRefIsValid( mLDAP ) )
		{
			if (!mCanceled)
			{
				(void) PGPldapGetErrno( mLDAP, NULL, &szError, &result );
				SetErrorString( szError );
			}
		}

		if (mCanceled)
		{
			mCanceled = false;
			ThrowPGPError_(kPGPError_UserAbort);
		} 
		else 
			throw;
	}
}



	void
CLDAPKeyServer::Close()
{
	StPreserveSocketsEventHandler	preserve(this);

	(void) pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Closing);
						
	delete[] mBaseKeySpaceDN;
	mBaseKeySpaceDN = 0;
	delete[] mBasePendingDN;
	mBasePendingDN = 0;
	if( PGPldapContextRefIsValid( mLDAP ) )
	{
		(void) PGPldapUnbind( mLDAP );
		mSecured = false;

		(void) PGPFreeLDAPContext( mLDAP );
		mLDAP = kInvalidPGPldapContextRef;
	}
	CKeyServer::Close();
}



	void
CLDAPKeyServer::Cancel()
{
	CKeyServer::Cancel();

	if( !PGPldapContextRefIsValid( mLDAP ) )
		return;

	(void) PGPldapUnbind( mLDAP );
	mSecured = false;
	
	(void) PGPFreeLDAPContext( mLDAP );
	mLDAP = kInvalidPGPldapContextRef;
}



	void
CLDAPKeyServer::Query(
	PGPFilterRef	inFilterRef,
	PGPKeyDBRef *	outFoundKeys)
{
	StPreserveSocketsEventHandler	preserve(this);
	char *							query = 0;
	PGPldapMessageRef				message = kInvalidPGPldapMessageRef;
	char **							value = 0;
	PGPKeyDBRef						foundKeys = kInvalidPGPKeyDBRef;
	PGPKeyDBRef						singleKeyDB = kInvalidPGPKeyDBRef;
	PGPKeySetRef					singleKeySet = kInvalidPGPKeySetRef;
	PGPBoolean						partialResults = false;
	PGPBoolean						receivedBadKeys = false;
	PGPldapResult					ldapResult 	= kPGPldapResult_Success;
	PGPError						pgpError 	= kPGPError_NoErr;
	PGPldapMessageRef				nextMessage = kInvalidPGPldapMessageRef;
	char *							szError		= NULL;
	
	SetErrorString( NULL );
	if (! mIsOpen)
		ThrowPGPError_(kPGPError_ServerNotOpen);
	
	try {
		char *				attribute;
		PGPberElementRef	iterAttribute;
		const char *		attr[2] =
								{ (mExportFormat == kPGPExportFormat_Basic) ?
									kPGPKeyAttribute : kPGPKeyV2Attribute,
										0 };

		pgpError = PGPLDAPQueryFromFilter(inFilterRef, &query);
		ThrowIfPGPErrorOrLDAPCanceled_(pgpError);

		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Querying);
		ThrowIfPGPErrorOrLDAPCanceled_(pgpError);
		
		// Send the query
		
		pgpError = PGPNewLDAPMessage( mLDAP, &message );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		pgpError = PGPldapSearchSync(	mLDAP,
							( mKeySpace == kPGPKeyServerKeySpace_Pending ) ?
								mBasePendingDN : 
								mBaseKeySpaceDN,
							kPGPldapScope_Subtree,
							query,
							(char **) attr,
							FALSE,
							message );

		ThrowIfLDAPCanceled_();

		switch( pgpError )
		{
			case kPGPError_NoErr:
				break;
			
			case kPGPError_LDAPInsufficientAccess:
				ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
				break;
			
			case kPGPError_LDAPSizelimitExceeded:
			case kPGPError_LDAPTimelimitExceeded:
				partialResults = true;
				break;
			
			default:
				ThrowPGPError_(kPGPError_ServerSearchFailed);
				break;
		}
		PGPFreeData(query);
		query = NULL;
		
		// Get the results
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_ProcessingResults);
		ThrowIfPGPErrorOrLDAPCanceled_(pgpError);
		pgpError = PGPNewKeyDB(mContext, &foundKeys);
		ThrowIfPGPErrorOrLDAPCanceled_(pgpError);

		pgpError = PGPldapFirstEntry( mLDAP, message, &nextMessage );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		while( PGPldapMessageRefIsValid( nextMessage ) )
		{
			pgpError = PGPldapFirstAttribute( mLDAP, nextMessage, &iterAttribute, 
				&attribute );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );

			while( IsntNull( attribute ) )
			{
				pgpError = PGPldapGetValues( mLDAP, nextMessage, 
					attribute,
					&value );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );

				if ((value != 0) && (*value != 0))
				{
					if (pgpCompareStringsIgnoreCase(attribute, attr[0]) == 0)
					{
						pgpError = PGPImport(
										mContext,
										&singleKeyDB,
										PGPOInputBuffer(	mContext,
															*value,
															strlen( *value ) ),
										PGPOLastOption( mContext ) );
						if( IsPGPError( pgpError ) )
							receivedBadKeys = true;
						else 
						{
							pgpError = PGPNewKeySet( singleKeyDB, &singleKeySet );
							ThrowIfPGPError_( pgpError );
							
							pgpError = PGPCopyKeys( singleKeySet, foundKeys, NULL );
							ThrowIfPGPError_( pgpError );
							
							PGPFreeKeySet( singleKeySet );
							singleKeySet = kInvalidPGPKeySetRef;
							
							PGPFreeKeyDB( singleKeyDB );
							singleKeyDB = kInvalidPGPKeyDBRef;
						}
					}
					(void) PGPFreeLDAPValues( value );
					value = NULL;
				}
				else
				{
					pgpError = PGPldapGetErrno( mLDAP, NULL, NULL, &ldapResult );
					if( IsPGPError( pgpError ) )
						ThrowPGPError_( kPGPError_ServerSearchFailed );

					if ( ( IsNull( value ) ) && ( ldapResult != kPGPldapResult_Success ) )
						ThrowPGPError_( kPGPError_ServerSearchFailed );
				}

				PGPFreeData( attribute );
				attribute = NULL;

				pgpError = PGPldapNextAttribute( mLDAP, nextMessage, iterAttribute, 
					&attribute );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );
			}

			pgpError = PGPldapNextEntry( mLDAP, nextMessage, &nextMessage );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );
		}

		PGPFreeLDAPMessage( message );
		message = kInvalidPGPldapMessageRef;

		*outFoundKeys = foundKeys;
	}
	
	catch (...) {
		if( singleKeySet != kInvalidPGPKeySetRef )
			PGPFreeKeySet( singleKeySet );

		if( singleKeyDB != kInvalidPGPKeyDBRef )
			PGPFreeKeyDB( singleKeyDB );

		if( foundKeys != kInvalidPGPKeyDBRef )
			PGPFreeKeyDB(foundKeys);

		if( value != 0 )
			PGPFreeLDAPValues( value );

		if( PGPldapMessageRefIsValid( message ) )
			PGPFreeLDAPMessage( message );

		if( query != 0 )
			PGPFreeData( query );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}
	
	if( receivedBadKeys )
		ThrowPGPError_( kPGPError_ServerBadKeysInSearchResults );

	if( partialResults )
		ThrowPGPError_( kPGPError_ServerPartialSearchResults );

}



	void
CLDAPKeyServer::Upload(
	PGPKeyUploadPreference inSendPrivateKeys,
	PGPKeySetRef	inKeysToUpload,
	PGPKeySetRef *	outKeysThatFailed)
{
	StPreserveSocketsEventHandler	preserve(this);
	
	char *			keySpace				= NULL;
	char *			exportedKey				= NULL;
	PGPKeySetRef	errorKeySet				= kInvalidPGPKeySetRef;
	PGPKeyListRef	keyList					= kInvalidPGPKeyListRef;
	PGPKeyIterRef	keyIter					= kInvalidPGPKeyIterRef;
	PGPKeySetRef	singleKeySet			= kInvalidPGPKeySetRef;
	PGPBoolean		sendRequest				= false;
	PGPBoolean		authorizationFailed		= false;
	PGPBoolean		partialFailure			= false;
	PGPBoolean		keyAlreadyExists		= false;
	PGPBoolean		failedPolicy			= false;
	PGPError		pgpError				= kPGPError_NoErr;
	char *			keySpaceSuffix			= NULL;
	PGPBoolean		exportPrivateKeys		= false;
	char *			szError					= NULL;
	PGPldapResult	ldapResult				= kPGPldapResult_Success;
	
	if( mKeySpace == kPGPKeyServerKeySpace_Pending )
		keySpaceSuffix = mBasePendingDN;
	else
		keySpaceSuffix = mBaseKeySpaceDN;
	
	SetErrorString( NULL );
	if( !mIsOpen )
		ThrowPGPError_(kPGPError_ServerNotOpen);

	if( inSendPrivateKeys == kPGPPrivateKeyAllowed )
	{
		if( mType != kPGPKeyServerProtocol_LDAPS )
			ThrowPGPError_( kPGPError_ServerOperationRequiresTLS );

		exportPrivateKeys = true;
	}

	try
	{
		PGPKeyDBObjRef		key;
		PGPSize				bufSize;
		char *				strVals[2] = { 0, 0 };
		PGPldapMod			mod = { kPGPldapModOp_Add,
									( ( mExportFormat == kPGPExportFormat_Basic ) ?
										(char *) kPGPKeyAttribute : 
										(char *) kPGPKeyV2Attribute ),
									strVals,
									(PGPberValue **) NULL };
		PGPldapMod *		attrs[2] = { &mod, NULL };

		
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Uploading);
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
			
		// Create error set
		pgpError = PGPNewEmptyKeySet( PGPPeekKeySetKeyDB( inKeysToUpload ), &errorKeySet );
		ThrowIfPGPError_( pgpError );
		
		// Create keyspace
		keySpace = new char[strlen( kAddAndRequestDNPrefix ) + 
							strlen(	keySpaceSuffix ) + 1];
		if( IsNull( keySpace ) )
			ThrowPGPError_( kPGPError_OutOfMemory );

		sprintf( keySpace, "%s%s", kAddAndRequestDNPrefix, keySpaceSuffix );
		
		// Iterate through keys uploading them one at a time
		pgpError = PGPOrderKeySet(	inKeysToUpload,
									kPGPKeyOrdering_Any,
									FALSE,
									&keyList );
		ThrowIfPGPError_( pgpError );
		pgpError = PGPNewKeyIter( keyList, &keyIter );
		ThrowIfPGPError_( pgpError);
		while( ( pgpError = PGPKeyIterNextKeyDBObj( keyIter, 
								kPGPKeyDBObjType_Key,
								&key ) ) == kPGPError_NoErr )
		{
			// Send to server
			if( !sendRequest )
			{
				pgpError = PGPExport( mContext,
								PGPOExportKeyDBObj( mContext, key ),
								PGPOExportPrivateKeys(mContext,
													exportPrivateKeys),
								PGPOAllocatedOutputBuffer(
													mContext,
													(void **) &exportedKey,
													MAX_PGPSize,
													&bufSize),
								PGPOExportFormat(mContext,
												mExportFormat),
								PGPOLastOption(mContext));

				ThrowIfPGPError_( pgpError );
				strVals[0] = exportedKey;
				
				pgpError = PGPldapAddSync( mLDAP, keySpace, attrs );
				if( IsPGPError( pgpError ) )
				{
					partialFailure = true;
	
					switch ( pgpError )
					{
						case kPGPError_LDAPAlreadyExists:
							keyAlreadyExists = true;
							break;
						
						case kPGPError_LDAPInsufficientAccess:
							sendRequest = true;
							break;
						
						case kPGPError_LDAPInvalidCredentials:
							failedPolicy = true;
							break;

						default:
							ThrowIfPGPError_( pgpError );
							break;
					}
				}

				pgpError = PGPAddKey(key, errorKeySet);
				ThrowIfPGPError_( pgpError );

				PGPFreeData( exportedKey);
				exportedKey = 0;
			}
			
			// Need to send a signed request
			if( sendRequest && ( !mSecured ) )
			{
				pgpError = PGPNewOneKeySet( key, &singleKeySet );
				ThrowIfPGPError_( pgpError );

				pgpError = SendRequest( kAdd, singleKeySet, 0 );
				if( IsPGPError( pgpError ) ) {
					partialFailure = true;
					(void) PGPAddKeys( singleKeySet, errorKeySet );

					switch( pgpError )
					{
						case kPGPError_ServerKeyAlreadyExists:
							keyAlreadyExists = true;
							break;
						
						case kPGPError_ServerAuthorizationFailed:
							authorizationFailed = true;
							break;
						
						case kPGPError_ServerKeyFailedPolicy:
							failedPolicy = true;
							break;
					}
				}

				PGPFreeKeySet(singleKeySet);
				singleKeySet = kInvalidPGPKeySetRef;
			}
		}

		delete[] keySpace;
		keySpace = 0;

		PGPFreeKeyIter( keyIter );
		keyIter = kInvalidPGPKeyIterRef;

		PGPFreeKeyList( keyList );
		keyList = kInvalidPGPKeyListRef;

		if( partialFailure )
		{
			*outKeysThatFailed = errorKeySet;
			errorKeySet = kInvalidPGPKeySetRef;

			if (failedPolicy)
				ThrowPGPError_(kPGPError_ServerKeyFailedPolicy);
			else if (authorizationFailed)
				ThrowPGPError_(kPGPError_ServerAuthorizationFailed);
			else if (keyAlreadyExists)
				ThrowPGPError_(kPGPError_ServerKeyAlreadyExists);
			else
				ThrowPGPError_(kPGPError_ServerPartialAddFailure);
		}
		else
		{
			PGPFreeKeySet(errorKeySet);
			errorKeySet = kInvalidPGPKeySetRef;
		}
	}
	
	catch (...)
	{
		delete[] keySpace;
		if (exportedKey != 0)
			PGPFreeData(exportedKey);

		if (singleKeySet != kInvalidPGPKeySetRef)
			PGPFreeKeySet(singleKeySet);

		if (keyIter != kInvalidPGPKeyIterRef)
			PGPFreeKeyIter(keyIter);

		if (keyList != kInvalidPGPKeyListRef)
			PGPFreeKeyList(keyList);

		if (errorKeySet != kInvalidPGPKeySetRef)
			PGPFreeKeySet(errorKeySet);

		if (mCanceled)
			ThrowPGPError_(kPGPError_UserAbort);
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}
}



	void
CLDAPKeyServer::Delete(
	PGPKeySetRef	inKeysToDelete,
	PGPKeySetRef *	outKeysThatFailed)
{
	DeleteOrDisable(true, inKeysToDelete, outKeysThatFailed);
}



	void
CLDAPKeyServer::Disable(
	PGPKeySetRef	inKeysToDisable,
	PGPKeySetRef *	outKeysThatFailed)
{
	DeleteOrDisable(false, inKeysToDisable, outKeysThatFailed);
}



	void
CLDAPKeyServer::DeleteOrDisable(
	PGPBoolean		inDelete,
	PGPKeySetRef	inKeySet,
	PGPKeySetRef *	outKeysThatFailed)
{
	StPreserveSocketsEventHandler	preserve(this);
	const char *	action		= 0;
	char *			szError		= NULL;
	PGPldapResult	result		= kPGPldapResult_Success;
	PGPError		pgpError	= kPGPError_NoErr;

	SetErrorString( NULL );
	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try 
	{
		PGPKeyServerState	state;
		
		if( mAccessType != kPGPKeyServerAccessType_Administrator )
			ThrowPGPError_( kPGPError_ServerAuthorizationRequired );

		if( inDelete )
		{
			action = kDelete;
			state = kPGPKeyServerState_Deleting;
		} 
		else
		{
			action = kDisable;
			state = kPGPKeyServerState_Disabling;
		}
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this, 
										state );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
		pgpError = SendRequest(	action,
								inKeySet,
								outKeysThatFailed );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
	}
	
	catch (...)
	{
		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &result );
			SetErrorString( szError );
			throw;
		}
	}
}



	PGPError
CLDAPKeyServer::SendRequest(
	const char *	inAction,
	PGPKeySetRef	inKeySetRef,
	PGPKeySetRef *	outKeysThatFailed)
{
	PGPError		result			= kPGPError_NoErr;
	char *			request			= NULL;
	char *			signedRequest	= NULL;
	PGPKeyListRef	keyList			= kInvalidPGPKeyListRef;
	PGPKeyIterRef	keyIter			= kInvalidPGPKeyIterRef;
	char *			keySpace		= NULL;
	char *			error			= NULL;
	char *			szError			= NULL;
	
	try
	{
		PGPUInt32			requestSize		= 0;
		PGPKeyID			keyID;
		char				keyIDString[kPGPMaxKeyIDStringSize];
		const char *		location		= NULL;
		PGPUInt32			total;
		PGPError			pgpError		= kPGPError_NoErr;
		PGPOptionListRef	options;
		PGPKeyDBObjRef		key;
		PGPSize				dummy;
		char *				strVals[2] = { 0, 0 };
		PGPldapMod			mod = { kPGPldapModOp_Add,
									(char *) kRequestPGPRequest,
									strVals,
									(PGPberValue **) NULL };
		PGPldapMod *		attrs[2] = { &mod, 0 };
		PGPldapResult		ldapResult		= kPGPldapResult_Success;
		char *				nextItem		= NULL;
		
		if( mKeySpace == kPGPKeyServerKeySpace_Pending )
			location = kRequestPending;
		else
			location = kRequestActive;
		
		// Allocate request buffer and add prefix
		pgpError = PGPCountKeys( inKeySetRef, &total );
		ThrowIfPGPError_( pgpError );
		request = new char[strlen( kRequestRequest ) + strlen( inAction )
							+ strlen( kRequestLocation ) + strlen( location )
							+ ( ( kPGPMaxKeyIDStringSize + 2 ) * total ) + 7];
		if( request == 0 )
			ThrowPGPError_(kPGPError_OutOfMemory);

		requestSize = sprintf(	request,
								"%s%s\r\n%s%s\r\n\r\n",
								kRequestRequest,
								inAction,
								kRequestLocation,
								location );
		
		// Iterate through keys creating the request
		pgpError = PGPOrderKeySet( inKeySetRef, kPGPKeyOrdering_Any, FALSE, &keyList );
		ThrowIfPGPError_( pgpError );

		pgpError = PGPNewKeyIter( keyList, &keyIter );
		ThrowIfPGPError_(pgpError);

		while( ( pgpError = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Key, &key) )
					== kPGPError_NoErr)
		{
			pgpError = PGPGetKeyID( key, &keyID );
			ThrowIfPGPError_( pgpError );

			pgpError = PGPGetKeyIDString(	&keyID,
											kPGPKeyIDString_Full,
											keyIDString );
			ThrowIfPGPError_( pgpError );

			requestSize += sprintf(	request + requestSize,
									"%s\r\n",
									keyIDString );
		}
		PGPFreeKeyIter( keyIter );
		keyIter = kInvalidPGPKeyIterRef;

		PGPFreeKeyList( keyList );
		keyList = kInvalidPGPKeyListRef;

		if( pgpError != kPGPError_EndOfIteration )
			ThrowIfPGPError_( pgpError );

		// Now sign the request if we do not have a secure connection
		if( ! mSecured )
		{
			do
			{
				pgpError = pgpEventKeyServerSign(	mContext,
													&options,
													mEventHandler,
													mEventHandlerData,
													(PGPKeyServerRef) this );
				ThrowIfPGPErrorOrLDAPCanceled_( pgpError );

				pgpError = PGPEncode(	mContext,
										PGPOInputBuffer( mContext,
													request,
													requestSize ),
										PGPOAllocatedOutputBuffer(
													mContext,
													(void**) &signedRequest,
													MAX_PGPSize,
													&dummy ),
										options,
										PGPOLastOption( mContext ) );
			} while (pgpError == kPGPError_BadPassphrase);

			ThrowIfPGPError_(pgpError);

			delete[] request;
			request = NULL;
		}
		
		// Send the request
		mod.value[0] = ( mSecured ? request : signedRequest );

		keySpace = new char[strlen( kAddAndRequestDNPrefix )
							+ strlen( mBaseKeySpaceDN ) + 1];
		if( keySpace == 0 )
			ThrowPGPError_( kPGPError_OutOfMemory );

		sprintf( keySpace, "%s%s", kAddAndRequestDNPrefix, mBaseKeySpaceDN );

		pgpError = PGPldapAddSync( mLDAP, keySpace, attrs );

		switch( pgpError )
		{
			case kPGPError_NoErr:
			case kPGPError_LDAPAlreadyExists:
			case kPGPError_LDAPInsufficientAccess:
			case kPGPError_LDAPInvalidCredentials:
				/* Ignore these errors for now.  We'll handle them later. */
				break;
			
			default:
				ThrowPGPError_( kPGPError_ServerRequestFailed );
				break;
		}

		ThrowIfLDAPCanceled_();

		delete[] keySpace;
		keySpace = NULL;

		if( mSecured )
		{
			delete[] request;
			request = 0;
		}
		else
		{
			PGPFreeData(signedRequest);
			signedRequest = 0;
		}
	
		// Scan for error keys
		if( ( outKeysThatFailed != 0 ) && ( IsntNull( szError ) ) &&
			( *szError != '\0' ) )
		{
			char *	currentItem = NULL;
		
			error = new char[strlen( szError ) + 1];

			if( IsNull( error ) )
				ThrowPGPError_(kPGPError_OutOfMemory);

			strcpy( error, szError );
			currentItem = strchr( error, ':' );

			if( currentItem != 0 )
			{
				currentItem++;
				while( ( *currentItem != '\0' ) && isspace( *currentItem ) )
					currentItem++;

				while( *currentItem != '\0' )
				{
					nextItem = currentItem;
					while( ( *nextItem != '\0' ) && ( !isspace( *nextItem ) ) )
						nextItem++;

					if( *nextItem != '\0' )
						*nextItem++ = '\0';

					pgpError = PGPNewKeyIDFromString( currentItem,
									kPGPPublicKeyAlgorithm_Invalid,
									&keyID);
					ThrowIfPGPError_(pgpError);

					pgpError = PGPFindKeyByKeyID(
									PGPPeekKeySetKeyDB( inKeySetRef ),
									&keyID,
									&key);
					ThrowIfPGPError_(pgpError);

					if( *outKeysThatFailed == kInvalidPGPKeySetRef )
					{
						pgpError = PGPNewEmptyKeySet( PGPPeekKeySetKeyDB( inKeySetRef ),
										outKeysThatFailed );
						ThrowIfPGPError_( pgpError );
					}

					pgpError = PGPAddKey( key, *outKeysThatFailed );
					ThrowIfPGPError_( pgpError );

					currentItem = nextItem;
					while( ( *currentItem != '\0' ) && isspace( *currentItem ) )
						currentItem++;
				}
			}

			delete[] error;
			error = NULL;
		}
			
		pgpError = PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
		ThrowIfPGPError_( pgpError );

		switch( ldapResult )
		{
			case kPGPldapResult_Success:
				break;
			
			case kPGPldapResult_AlreadyExists:
				ThrowPGPError_( kPGPError_ServerKeyAlreadyExists );
				break;
			
			case kPGPldapResult_InsufficientAccess:
				ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
				break;
			
			case kPGPldapResult_InvalidCredentials:
				ThrowPGPError_( kPGPError_ServerKeyFailedPolicy );
				break;
			
			default:
				ThrowPGPError_( kPGPError_ServerRequestFailed );
				break;
		}
	}
	
	catch( PGPError exception )
	{
		delete[] error;
		delete[] request;
		delete[] keySpace;

		if( IsntNull( signedRequest ) )
			PGPFreeData( signedRequest );

		if( PGPKeyIterRefIsValid( keyIter ) )
			PGPFreeKeyIter( keyIter );

		if( PGPKeyListRefIsValid( keyList ) )
			PGPFreeKeyList( keyList );

		result = exception;
	}
	
	return result;
}



	void
CLDAPKeyServer::SendGroups(
	PGPGroupSetRef	inGroupSetRef)
{
	StPreserveSocketsEventHandler	preserve(this);
	void *							groupsBuffer	= 0;
	PGPError						pgpError		= kPGPError_NoErr;
	PGPldapResult					ldapResult		= kPGPldapResult_Success;
	char *							szError			= NULL;
	
	SetErrorString( NULL );
	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		PGPberValue			berVal = { 0, 0 };
		PGPberValue *		berVals[2] = { &berVal, NULL };
		const char *		strVals[2] = { kPGPSrvGroupFileClass, 0 };
		PGPldapMod			mod = { (PGPldapModOp) 
									(kPGPldapModOp_Add | kPGPldapModOpMask_UseBERValues),
								(char *) kPGPSrvGroupFileV2,
								(char **) NULL,
								(PGPberValue **) berVals };
		PGPldapMod *		attrs[2] = { &mod, NULL };
		PGPSize				groupsBufferSize;
		
		
		// Export the groups
		pgpError = PGPExportGroupSetToBuffer(	inGroupSetRef,
												&groupsBuffer,
												&groupsBufferSize );
		ThrowIfPGPError_( pgpError );
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Uploading );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
		
		// Send them to the server
		berVals[0]->length = groupsBufferSize;
		berVals[0]->value = (PGPByte *) groupsBuffer;

		pgpError = PGPldapAddSync( mLDAP, (char *) kDefaultPGPGroupspace, attrs );

		ThrowIfLDAPCanceled_();

		if( IsPGPError( pgpError ) )
		{
			switch( pgpError )
			{
				case kPGPError_LDAPAlreadyExists:
					ThrowPGPError_( kPGPError_ServerKeyAlreadyExists );
					break;
				
				case kPGPError_LDAPInsufficientAccess:
					ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
					break;
				
				case kPGPError_LDAPInvalidCredentials:
					ThrowPGPError_( kPGPError_ServerKeyFailedPolicy );
					break;
				
				default:
					ThrowPGPError_( kPGPError_ServerRequestFailed );
					break;
			}
		}
		PGPFreeData( groupsBuffer );
		groupsBuffer = NULL;
		
		// Add the objectclass
		mod.op = kPGPldapModOp_Add;
		mod.type = (char *) kObjectClass;
		mod.value = (char **) strVals;

		pgpError = PGPldapAddSync( mLDAP, (char *) kDefaultPGPGroupspace, attrs );

		ThrowIfLDAPCanceled_();

		if( IsPGPError( pgpError ) )
		{
			switch( pgpError )
			{
				case kPGPError_LDAPAlreadyExists:
					ThrowPGPError_( kPGPError_ServerKeyAlreadyExists );
					break;
				
				case kPGPError_LDAPInsufficientAccess:
					ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
					break;
				
				case kPGPError_LDAPInvalidCredentials:
					ThrowPGPError_( kPGPError_ServerKeyFailedPolicy );
					break;
				
				default:
					ThrowPGPError_( kPGPError_ServerRequestFailed );
					break;
			}
		}
	}
	
	catch (...)
	{
		if( IsntNull( groupsBuffer ) )
			PGPFreeData( groupsBuffer );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}
}



	void
CLDAPKeyServer::RetrieveGroups(
	PGPGroupSetRef *	outGroupSetRef)
{
	StPreserveSocketsEventHandler	preserve(this);
	PGPldapMessageRef				message		= kInvalidPGPldapMessageRef;
	PGPberValue **					value		= NULL;
	PGPGroupSetRef					groupSetRef = kInvalidPGPGroupSetRef;
	PGPError						pgpError	= kPGPError_NoErr;
	PGPldapResult					ldapResult	= kPGPldapResult_Success;
	char *							szError		= NULL;

	SetErrorString( NULL );

	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		PGPldapMessageRef	nextMessage		= kInvalidPGPldapMessageRef;
		char *				attribute		= NULL;
		PGPberElementRef	iterAttribute	= kInvalidPGPberElementRef;
		static const char *	attr[2] = { kPGPSrvGroupFileV2, 0 };
		
		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Querying );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );
		
		pgpError = PGPNewLDAPMessage( mLDAP, &message );
		ThrowIfPGPError_( pgpError );

		pgpError = PGPldapSearchSync(  mLDAP,
							(char *) kDefaultPGPGroupspace,
							kPGPldapScope_Base,
							(char *) kObjectClassAny,
							(char **) attr,
							FALSE,
							message );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		pgpError = PGPldapFirstEntry( mLDAP, message, &nextMessage );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		if( PGPldapMessageRefIsValid( nextMessage ) )
		{
			pgpError = PGPldapFirstAttribute( mLDAP, nextMessage, &iterAttribute, 
				&attribute );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );

			if( IsntNull( attribute ) )
			{
				pgpError = PGPldapGetValuesLen( mLDAP, 
					nextMessage, 
					attribute,
					&value );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );

				if( ( value != 0 ) && ( *value != 0 ) )
				{
					pgpError = PGPImportGroupSetFromBuffer(	mContext,
															(*value)->value,
															(*value)->length,
															&groupSetRef );
					if( IsPGPError( pgpError ) )
						ThrowPGPError_( kPGPError_ServerSearchFailed );

					PGPFreeLDAPValuesLen( value );
					value = NULL;
				}
			}
		}
		if( groupSetRef == kInvalidPGPGroupSetRef )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		PGPFreeLDAPMessage( message );
		message = kInvalidPGPldapMessageRef;

		*outGroupSetRef = groupSetRef;
	}
	
	catch(...)
	{
		if( groupSetRef != kInvalidPGPGroupSetRef )
			PGPFreeGroupSet( groupSetRef );

		if( IsntNull( value ) )
			PGPFreeLDAPValuesLen( value );

		if( PGPldapMessageRefIsValid( message ) )
			PGPFreeLDAPMessage( message );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}
}



	void
CLDAPKeyServer::NewMonitor(
	PGPKeyServerMonitor **	outMonitor)
{
	StPreserveSocketsEventHandler	preserve(this);
	PGPldapMessageRef				message				= kInvalidPGPldapMessageRef;
	char **							value				= NULL;
	PGPKeyServerMonitor *			monitor				= 0;
	PGPError						pgpError			= kPGPError_NoErr;
	PGPldapResult					ldapResult			= kPGPldapResult_Success;
	PGPKeyServerMonitorValues *		currentMonitorValue	= 0;
	PGPldapMessageRef				nextMessage			= kInvalidPGPldapMessageRef;
	char *							attribute			= NULL;
	int								numValues			= 0;
	int 							i					= 0;
	char *							szError				= NULL;

	SetErrorString( NULL );
	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		PGPberElementRef			iterAttribute;
		static const char *			attr[1] = { 0 };

		if( mAccessType != kPGPKeyServerAccessType_Administrator )
			ThrowPGPError_( kPGPError_ServerAuthorizationRequired );

		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_Querying );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );

		PGPNewLDAPMessage( mLDAP, &message );

		pgpError = PGPldapSearchSync(	mLDAP,
							(char *) kMonitorCN,
							kPGPldapScope_Base,
							(char *) kObjectClassAny,
							(char **) attr,
							FALSE,
							message );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		pgpError = pgpEventKeyServer(	mContext,
										mEventHandler,
										mEventHandlerData,
										(PGPKeyServerRef) this,
										kPGPKeyServerState_ProcessingResults );
		ThrowIfPGPErrorOrLDAPCanceled_( pgpError );

		monitor = new PGPKeyServerMonitor;
		if( IsNull( monitor ) )
			ThrowPGPError_( kPGPError_OutOfMemory );

		monitor->keyServerRef = (PGPKeyServerRef) this;
		monitor->valuesHead = 0;

		pgpError = PGPldapFirstEntry( mLDAP, message, &nextMessage );
		if( IsPGPError( pgpError ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		while( PGPldapMessageRefIsValid( nextMessage ) )
		{
			pgpError = PGPldapFirstAttribute(  mLDAP,
									nextMessage,
									&iterAttribute,
									&attribute );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );

			while( IsntNull( attribute ) )
			{
				if( IsNull( currentMonitorValue ) )
				{
					monitor->valuesHead = currentMonitorValue = 
											new PGPKeyServerMonitorValues;
				}
				else
				{
					currentMonitorValue->next =
											new PGPKeyServerMonitorValues;
					currentMonitorValue = currentMonitorValue->next;
				}
				if( IsNull( currentMonitorValue ) )
					ThrowPGPError_( kPGPError_OutOfMemory );

				currentMonitorValue->values = 0;
				currentMonitorValue->next = 0;
				currentMonitorValue->name = new char[strlen(attribute) + 1];

				if( IsNull( currentMonitorValue->name ) )
					ThrowPGPError_( kPGPError_OutOfMemory );

				/* copy the attribute */
				strncpy( currentMonitorValue->name, attribute,
						strlen(attribute) + 1);
				
				pgpError = PGPldapGetValues( mLDAP, 
					nextMessage, 
					attribute, 
					&value );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );

				if( IsntNull( value ) )
				{
					pgpError = PGPldapCountValues( mLDAP, 
						value, 
						(PGPSize *) &numValues );
					if( IsPGPError( pgpError ) )
						ThrowPGPError_( kPGPError_ServerSearchFailed );

					currentMonitorValue->values = new char*[numValues + 1];
					if( IsNull( currentMonitorValue->values ) )
						ThrowPGPError_(kPGPError_OutOfMemory);

					for( i = 0; i < numValues; i++ )
					{
						currentMonitorValue->values[i] =
											new char[strlen(value[i]) + 1];
						if( IsNull( currentMonitorValue->values[i] ) )
							ThrowPGPError_(kPGPError_OutOfMemory);

						strcpy(currentMonitorValue->values[i], value[i]);
					}
					currentMonitorValue->values[numValues] = 0;
					PGPFreeLDAPValues( value );
					value = NULL;
				}
				PGPFreeData( attribute );
				attribute = NULL;

				pgpError = PGPldapNextAttribute( mLDAP, nextMessage, iterAttribute, 
					&attribute );
				if( IsPGPError( pgpError ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );
			}

			
			if( ldapResult != kPGPldapResult_Success )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			pgpError = PGPldapNextEntry( mLDAP, nextMessage, &nextMessage );
			if( IsPGPError( pgpError ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );
		}
		PGPFreeLDAPMessage( message );
		message = kInvalidPGPldapMessageRef;

		*outMonitor = monitor;
	}
	
	catch (...)
	{
		if( IsntNull( monitor ) )
			FreeMonitor( monitor );

		if( IsntNull( value ) )
			PGPFreeLDAPValues( value );

		if( PGPldapMessageRefIsValid( message ) )
			PGPFreeLDAPMessage( message );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}
}



	void
CLDAPKeyServer::FreeMonitor(
	PGPKeyServerMonitor *	inMonitor)
{
	PGPKeyServerMonitorValues *	currentMonitorValue = inMonitor->valuesHead;
	PGPKeyServerMonitorValues * tempMonitorValue;
	char **						tempValue;
	
	while( currentMonitorValue != 0 ) 
	{
		delete[] currentMonitorValue->name;

		if( currentMonitorValue->values != 0 )
		{
			tempValue = currentMonitorValue->values;
			while( *tempValue != 0 )
			{
				delete[] *tempValue;
				tempValue++;
			}
			delete[] currentMonitorValue->values;
		}

		tempMonitorValue = currentMonitorValue;
		currentMonitorValue = currentMonitorValue->next;

		delete tempMonitorValue;
	}

	delete[] inMonitor;
}
